using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents get user demographics response security message 
	/// defined in the VistALink protocol.
	/// </summary>
	public sealed class SecurityGetUserDemographicsResponseMessage : SecurityBaseResponseMessage
	{
		private readonly RemoteUserInfo _userInfo;
		private readonly DivisionInfo _divisionInfo;
		private readonly RemoteUserNameInfo _nameInfo;

		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "AV.GetUserDemographics";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secUserDemographicsResponse.xsd";

		/// <summary>
		/// Constructor creating an instance of class from supplied parameters.
		/// All parameters are required.
		/// </summary>
		/// <param name="nameInfo">Remote user name info.</param>
		/// <param name="userInfo">Remote user VistA info/</param>
		/// <param name="divisionInfo">Remote user division info.</param>
		public SecurityGetUserDemographicsResponseMessage( RemoteUserNameInfo nameInfo, RemoteUserInfo userInfo, DivisionInfo divisionInfo )
			: base( XsdName, ResponseType, ResponseStatus.Success )
		{
			if( nameInfo == null )
				throw( new ArgumentNullException( "nameInfo" ) );

			if( userInfo == null )
				throw( new ArgumentNullException( "userInfo" ) );

			if( divisionInfo == null )
				throw( new ArgumentNullException( "divisionInfo" ) );

			_nameInfo = nameInfo;
			_userInfo = userInfo;
			_divisionInfo = divisionInfo;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityGetUserDemographicsResponseMessage( XmlDocument sourceDoc )
			: base( XsdName, ResponseType, sourceDoc )
		{
			if( ResponseStatus != ResponseStatus.Success )
				throw( new XmlParseException( SR.Exceptions.InvalidResponseStatusFoundDuringDeserialization( 
					this.ResponseStatus.ToString(), this.GetType().Name, ResponseStatus.Success.ToString() ) ) );

			XmlElement _responseElement = BaseResponseMessage.XmlParseGetResponseElement( sourceDoc );

			_userInfo = RemoteUserInfo.ParseFindChildNodeByUniqueTagName( _responseElement );
			_nameInfo = RemoteUserNameInfo.ParseFindChildNodeByUniqueTagName( _responseElement );
			_divisionInfo = DivisionInfo.ParseFindChildNodeByUniqueTagName( _responseElement );
		}

		///<summary>
		/// Overriden method adding message-specific content to response element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			_nameInfo.WriteNameInfoToXml( writer );
			_userInfo.WriteUserInfoToXml( writer );
			_divisionInfo.WriteDivisionInfoToXml( writer );
		}

		/// <summary>
		/// Remote user VistA system info.
		/// </summary>
		public RemoteUserInfo UserInfo
		{
			get
			{
				return _userInfo;
			}
		}

		/// <summary>
		/// Remote user name info.
		/// </summary>
		public RemoteUserNameInfo UserNameInfo
		{
			get
			{
				return _nameInfo;
			}
		}

		/// <summary>
		/// User division info.
		/// </summary>
		public DivisionInfo UserDivisionInfo
		{
			get
			{
				return _divisionInfo;
			}
		}
	}
}
